#pragma once

static const float albedo_eps = 1e-10;

float2x4 compute_direct_light(
    uint max_bounces,
    Environment env,
    LocalGeometry lg,
    float3 out_dir,
    GltfShadeParams sp,
    bool cache_glossy,
    bool include_ambient
)
{
    const bool do_ms = is_ms_on(g_ubo.scene_flags);
    const bool do_nee = is_nee_on(g_ubo.scene_flags);
    const bool do_mis = is_mis_on(g_ubo.scene_flags);

    float4 direct_diffuse = float4(0, 0, 0, 1);
    float4 direct_glossy = float4(0, 0, 0, 1);

    bool has_throughput = sp.has_diffuse();
    if (cache_glossy)
        has_throughput = sp.has_diffuse() || sp.has_glossy() || sp.has_specular();

    // ambient
    if (include_ambient)
        direct_diffuse = float4(sp.emission + sp.lambertian_weight(0.0) * sp.base_color.rgb * (g_ubo.env.ambient), 0.0f);

    // cast shadow rays and compute contribution
    if (has_throughput && do_nee && !sp.is_specular() && all(sp.emission == 0) && g_ubo.max_bounces > 0)
    {
        LightSample ls = direct_lightsample(g_ubo.env, out_dir, sp, lg);
        float bsdf_pdf = 0.0f;
        if (cache_glossy)
            bsdf_pdf = gltf_pdf(sp, out_dir, ls.dir);
        else
            bsdf_pdf = gltf_diffuse_pdf(sp, out_dir, ls.dir);

        float w_ne = 1.0;
        bool mis_on = do_nee && do_mis && (g_ubo.max_bounces > 0);
        if (mis_on)
        {
            if (ls.environment)
                w_ne = mis_env(ls.pdf, bsdf_pdf);
            else if (ls.pdf > 0.0)
                w_ne = mis_lights(ls.pdf, bsdf_pdf);
        }

        float3 diffuse_brdf = eval_diffuse(sp, out_dir, ls.dir).rgb;
        direct_diffuse.rgb = w_ne * ls.intensity * diffuse_brdf;
        if (cache_glossy)
        {
            float3 glossy_brdf = eval_reflection(sp, out_dir, ls.dir).rgb;
            direct_glossy.rgb = w_ne * ls.intensity * glossy_brdf;
        }
    }

    return float2x4(direct_diffuse, direct_glossy);
}

float2x4 compute_indirect_light(
    uint max_bounces,
    Environment env,
    LocalGeometry lg,
    float3 out_dir,
    float coneDiff,
    float coneWidth,
    GltfShadeParams sp,
    bool cache_glossy
)
{
    const bool do_ms = is_ms_on(g_ubo.scene_flags);
    const bool do_nee = is_nee_on(g_ubo.scene_flags);
    const bool do_mis = is_mis_on(g_ubo.scene_flags);

    float4 indirect_diffuse = float4(0, 0, 0, 1);
    float4 indirect_glossy = float4(0, 0, 0, 1);
    
    if (!do_ms)
        float2x4(indirect_diffuse, indirect_glossy);

    bool has_throughput = sp.has_diffuse();
    if (cache_glossy)
        has_throughput = sp.has_diffuse() || sp.has_glossy() || sp.has_specular();

    if (has_throughput && g_ubo.max_bounces > 0 && (g_ubo.scene_flags & SCENE_FLAGS_MATSAMPLING) != 0)
    {
        float2 rnd_uv = float2(rnd(seed), rnd(seed));
        BsdfSample samp;
        if (cache_glossy)
        {
            sp.base_color.a = 1;
            samp = sample_gltf(sp, out_dir, rnd_uv);
        }
        else
        {
            samp = sample_diffuse(sp, out_dir, rnd_uv);
        }

        Ray r;
        r.origin = offset_ray(lg.pos, dot(lg.surface_normal, samp.inDir) > 0 ? lg.surface_normal : -lg.surface_normal);
        r.dir = samp.inDir;
        r.specular = samp.is_specular;
        r.coneDiff = coneDiff;
        r.coneWidth = coneWidth;
        r.tmax = TMAX;
        TraceState ts = init_tracestate(r, 1, g_ubo.max_bounces);
        ts.throughput = all(samp.reflectance > albedo_eps) && samp.pdf != 0 ? samp.reflectance / samp.pdf : 0;
        ts.brdf_pdf = samp.pdf;

        if (all(ts.throughput > 0))
        {
            TraceResult tr = trace_path(ts);
            indirect_diffuse.rgb = tr.radiance * eval_diffuse(sp, out_dir, samp.inDir).rgb / max(albedo_eps, samp.reflectance);

            if (cache_glossy)
            {
                indirect_glossy.rgb += tr.radiance * eval_reflection(sp, out_dir, samp.inDir).rgb / max(albedo_eps, samp.reflectance);
                indirect_glossy.rgb += tr.radiance * eval_transmission(sp, out_dir, samp.inDir).rgb / max(albedo_eps, samp.reflectance);
            }
        }
    }

    return float2x4(indirect_diffuse, indirect_glossy);
}
